#!/bin/bash

mysql_cust_cfg_file="/opt/caton/cydex3/custom/.cust_mysql"
redis_cust_cfg_file="/opt/caton/cydex3/custom/.cust_redis"
mongo_cust_cfg_file="/opt/caton/cydex3/custom/.cust_mongo"
rabbitmq_cust_cfg_file="/opt/caton/cydex3/custom/.cust_rabbitmq"

escape_slash() {
    msg=$1
    if [ ! -z "$msg" ]; then
        echo $msg | sed 's/\//\\\//g'
        return
    fi
    while read msg; do
        echo $msg | sed 's/\//\\\//g'
    done
}

encode_dsn_password() {
    echo $1 | sed 's/%/%25/g' | sed 's/@/%40/g'
}

decode_dsn_password() {
    echo $1 | sed 's/%40/@/g' | sed 's/%25/%/g'
}

load_cust_mysql() {
    local mysql_dsn="root:Caton_123@tcp(127.0.0.1:3306)"
    if [ -f $mysql_cust_cfg_file ]; then
        mysql_dsn=`cat $mysql_cust_cfg_file`
    fi
    if [ -z $mysql_dsn ]; then
        mysql_dsn="root:Caton_123@tcp(127.0.0.1:3306)"
    fi
    echo $mysql_dsn
}

load_cust_mysql_vars() {
    custom_mysql_username="root"
    custom_mysql_passwd="Caton_123"
    custom_mysql_host=""
    custom_mysql_port="3306"

    local mysql_dsn=$(load_cust_mysql)
    if [[ $mysql_dsn =~ ^([^:]*):([^@]*)@tcp\(([^:]*):([0-9]*)\)$ ]]; then
        custom_mysql_username="${BASH_REMATCH[1]}"
        custom_mysql_passwd=$(decode_dsn_password "${BASH_REMATCH[2]}")
        custom_mysql_host="${BASH_REMATCH[3]}"
        custom_mysql_port="${BASH_REMATCH[4]}"
    fi
}

run_mysql_script() {
    if [ -z "$custom_mysql_username" ] || [ -z "$custom_mysql_passwd" ]; then
        load_cust_mysql_vars
    fi
    mysql -u$custom_mysql_username -p$custom_mysql_passwd -h $custom_mysql_host -P $custom_mysql_port -e "$1"
}

set_cust_mysql() {
    if [ -z "$custom_mysql_username" ] || [ -z "$custom_mysql_passwd" ]; then
        load_cust_mysql_vars
    fi

    read -e -p " * Please specify username of custom MySQL server with all priviledges: " -i "$custom_mysql_username" custom_mysql_username
    while [ -z $custom_mysql_username ]; do
        read -e -p " * Please specify username of custom MySQL server with all priviledges: " -i "$custom_mysql_username" custom_mysql_username
    done

    read -e -p " * Please specify password of custom MySQL server: " -i "$custom_mysql_passwd" custom_mysql_passwd
    while [ -z $custom_mysql_passwd ]; do
        read -e -p " * Please specify password of custom MySQL server: " -i "$custom_mysql_passwd" custom_mysql_passwd
    done

    read -e -p " * Please specify host of custom MySQL server: " custom_mysql_host
    while [ -z $custom_mysql_host ]; do
        read -e -p " * Please specify host of custom MySQL server: " -i "$custom_mysql_host" custom_mysql_host
    done

    unset -v custom_mysql_port
    while [ -z $custom_mysql_port ]; do
        read -e -p " * Please specify port of custom MySQL server: " -i "3306" custom_mysql_port
    done

    local mysql_dsn="$custom_mysql_username:$(encode_dsn_password $custom_mysql_passwd)@tcp($custom_mysql_host:$custom_mysql_port)"

    mkdir -p `dirname $mysql_cust_cfg_file`
    echo $mysql_dsn > $mysql_cust_cfg_file

    mysql_dsn=`escape_slash $mysql_dsn`

    find /opt/caton/cydex3/ -type f \( -name *.conf -o -name *.ini \) -exec sed -i -re "s/=[^@]*@[^\/]*\/(cydex_admin|db_cydex|db_statistics|db_user|db_zone)/=$mysql_dsn\/\1/g" {} \;
}

mysql_db_exists() {
    if [ -z "$1" ]; then
        echo "DB name not provided"
        exit 1
    fi
    has_db=`run_mysql_script "SELECT * FROM information_schema.SCHEMATA where SCHEMA_NAME='$1';" | wc -l`
    [ ${has_db} -gt 0 ]
}

has_local_mysql() {
    [ `rpm -qa | grep mysql | wc -l` -gt 0 ]
}

has_local_mysql_server() {
    [ `rpm -qa | grep mysql | wc -l` -eq 5 ]
}

check_need_install_local_mysql_server() {
    # return false if explicitly indicated using external resource
    if [ "$1" == "1" ] ; then
        return 1
    fi
    if [ ! has_local_mysql_server ] && [ -f $mysql_cust_cfg_file ]; then
        echo "Skip installing local MySQL by default, since custom defined MySQL $(load_cust_mysql) found in $mysql_cust_cfg_file."
        read -e -p "Would you like to install local MySQL server anyway? (y/N)" answer
        if [ "$answer" == "Y" ] || [ "$answer" == "y" ]; then
            return 0
        fi
        return 1
    fi
    return 0
}

reset_local_mysql_password() {
    systemctl stop mysqld

    if [ ! -f /etc/my.cnf.backup ]; then
        mv /etc/my.cnf /etc/my.cnf.backup
    fi

    cp -r config/my.cnf /etc/my.cnf
    mv /var/log/mysqld.log /var/log/mysqld.log.backup >/dev/null 2>&1

    systemctl enable mysqld
    systemctl start mysqld

    sleep 3

    if [ -z "$custom_mysql_passwd" ]; then
        load_cust_mysql_vars
    fi
    # initial the root password
    ## get the random password
    orig_password=`cat /var/log/mysqld.log | grep "A temporary password" | awk -F "root@localhost: " '{print $2}'`
    mysqladmin -u root -p${orig_password} password $custom_mysql_passwd >/dev/null 2>&1
}

load_cust_redis() {
    local redis_dsn="redis://:Caton_123@127.0.0.1:6379"
    if [ -f $redis_cust_cfg_file ]; then
        redis_dsn=`cat $redis_cust_cfg_file`
    fi
    if [ -z $redis_dsn ]; then
        redis_dsn="redis://:Caton_123@127.0.0.1:6379"
    fi
    echo $redis_dsn
}

set_cust_redis() {
    local custom_username=""
    local custom_passwd="Caton_123"
    local custom_host=""
    local custom_port="6379"

    local redis_dsn=$(load_cust_redis)
    if [[ $redis_dsn =~ ^redis://([^:]*):([^@]*)@([^:]*):([0-9]*)$ ]]; then
        custom_username="${BASH_REMATCH[1]}"
        custom_passwd=$(decode_dsn_password "${BASH_REMATCH[2]}")
        custom_host="${BASH_REMATCH[3]}"
        custom_port="${BASH_REMATCH[4]}"
    fi
    
    read -e -p " * Please specify username of custom Redis server, can be empty: " -i "$custom_username" custom_username
    
    read -e -p " * Please specify password of custom Redis server: " -i "$custom_passwd" custom_passwd
    while [ -z $custom_passwd ]; do
        read -e -p " * Please specify password of custom Redis server: " -i "$custom_passwd" custom_passwd
    done

    read -e -p " * Please specify host of custom Redis server: " custom_host
    while [ -z $custom_host ]; do
        read -e -p " * Please specify host of custom Redis server: " -i "$custom_host" custom_host
    done

    read -e -p " * Please specify port of custom Redis server: " -i "$custom_port" custom_port
    while [ -z $custom_port ]; do
        read -e -p " * Please specify port of custom Redis server: " -i "$custom_port" custom_port
    done

    redis_dsn="redis://$custom_username:$(encode_dsn_password $custom_passwd)@$custom_host:$custom_port"

    mkdir -p `dirname $redis_cust_cfg_file`
    echo $redis_dsn > $redis_cust_cfg_file

    redis_dsn=`escape_slash $redis_dsn`
    find /opt/caton/cydex3/ -type f \( -name *.conf -o -name *.ini \) -exec sed -i -re "s/=redis:\/\/[^:]*:[^@]*@[^:]*:[^\/]*/=$redis_dsn/g" {} \;
}

has_local_redis_server() {
    [ `rpm -qa | grep redis | wc -l` -gt 0 ]
}

load_cust_mongo() {
    local mongo_dsn="mongodb://caton:Caton_123@127.0.0.1:27017"
    if [ -f $mongo_cust_cfg_file ]; then
        mongo_dsn=`cat $mongo_cust_cfg_file`
    fi
    if [ -z $mongo_dsn ]; then
        mongo_dsn="mongodb://caton:Caton_123@127.0.0.1:27017"
    fi
    echo $mongo_dsn
}

set_cust_mongo() {
    local custom_mongo_username="caton"
    local custom_mongo_passwd="Caton_123"
    local custom_mongo_host=""
    local custom_mongo_port="27017"

    local mongo_dsn=$(load_cust_mongo)
    if [[ $mongo_dsn =~ ^mongodb://([^:]*):([^@]*)@([^:]*):([0-9]*)$ ]]; then
        custom_mongo_username="${BASH_REMATCH[1]}"
        custom_mongo_passwd=$(decode_dsn_password "${BASH_REMATCH[2]}")
        custom_mongo_host="${BASH_REMATCH[3]}"
        custom_mongo_port="${BASH_REMATCH[4]}"
    fi

    read -e -p " * Please specify username of custom Mongo server: " -i "$custom_mongo_username" custom_mongo_username
    while [ -z $custom_mongo_username ]; do
        read -e -p " * Please specify username of custom Mongo server: " -i "$custom_mongo_username" custom_mongo_username
    done
    
    read -e -p " * Please specify password of custom Mongo server: " -i "$custom_mongo_passwd" custom_mongo_passwd
    while [ -z $custom_mongo_passwd ]; do
        read -e -p " * Please specify password of custom Mongo server: " -i "$custom_mongo_passwd" custom_mongo_passwd
    done

    read -e -p " * Please specify host of custom Mongo server: " custom_mongo_host
    while [ -z $custom_mongo_host ]; do
        read -e -p " * Please specify host of custom Mongo server: " -i "$custom_mongo_host" custom_mongo_host
    done

    read -e -p " * Please specify port of custom Mongo server: " -i "$custom_mongo_port" custom_mongo_port
    while [ -z $custom_mongo_port ]; do
        read -e -p " * Please specify port of custom Mongo server: " -i "$custom_mongo_port" custom_mongo_port
    done

    local mongo_dsn="mongodb://$custom_mongo_username:$(encode_dsn_password $custom_mongo_passwd)@$custom_mongo_host:$custom_mongo_port"

    mkdir -p `dirname $mongo_cust_cfg_file`
    echo $mongo_dsn > $mongo_cust_cfg_file

    sed -i -re "s/;? *uri=.*$/uri=$(escape_slash $mongo_dsn)/g" /opt/caton/cydex3/zone/config.ini
    sed -i -re "s/;? * username=.*$/; username=$custom_mongo_username/g" /opt/caton/cydex3/zone/config.ini
    sed -i -re "s/;? * password=.*$/; password=`escape_slash $custom_mongo_passwd`/g" /opt/caton/cydex3/zone/config.ini
}

has_local_mongo() {
    [ `rpm -qa | grep mongodb | wc -l` -gt 0 ]
}

has_local_mongo_server() {
    [ `rpm -qa mongodb-org-server | wc -l` -gt 0 ]
}

check_need_install_local_mongo_server() {
    # return false if explicitly indicated using external resource
    if [ "$1" == "1" ] ; then
        return 1
    fi
    if [ ! has_local_mongo_server ] && [ -f $mongo_cust_cfg_file ]; then
        echo "Skip installing local MongoDB by default, since custom defined MongoDB $(load_cust_mongo) found in $mongo_cust_cfg_file."
        read -e -p "Would you like to install local MongoDB server anyway? (y/N)" answer
        if [ "$answer" == "Y" ] || [ "$answer" == "y" ]; then
            return 0
        fi
        return 1
    fi
    return 0
}

load_cust_rabbitmq_vars() {
    custom_rabbitmq_username="caton"
    custom_rabbitmq_passwd="Caton_123"
    custom_rabbitmq_host=""
    custom_rabbitmq_port="5672"
    if [ -f $rabbitmq_cust_cfg_file ]; then
        local var_arr
        readarray -n 4 -t var_arr < $rabbitmq_cust_cfg_file
        custom_rabbitmq_username="${var_arr[0]}"
        custom_rabbitmq_passwd="${var_arr[1]}"
        custom_rabbitmq_host="${var_arr[2]}"
        custom_rabbitmq_port="${var_arr[3]}"
    fi
}

set_cust_rabbitmq() {
    if [ -z "$custom_rabbitmq_username" ] || [ -z "$custom_rabbitmq_passwd" ]; then
        load_cust_rabbitmq_vars
    fi

    read -e -p " * Please specify username of custom RabbitMQ server: " -i "$custom_rabbitmq_username" custom_rabbitmq_username
    while [ -z $custom_rabbitmq_username ]; do
        read -e -p " * Please specify username of custom RabbitMQ server: " -i "$custom_rabbitmq_username" custom_rabbitmq_username
    done
    
    read -e -p " * Please specify password of custom RabbitMQ server: " -i "$custom_rabbitmq_passwd" custom_rabbitmq_passwd
    while [ -z $custom_rabbitmq_passwd ]; do
        read -e -p " * Please specify password of custom RabbitMQ server: " -i "$custom_rabbitmq_passwd" custom_rabbitmq_passwd
    done

    read -e -p " * Please specify host of custom RabbitMQ server: " custom_rabbitmq_host
    while [ -z $custom_rabbitmq_host ]; do
        read -e -p " * Please specify host of custom RabbitMQ server: " -i "$custom_rabbitmq_host" custom_rabbitmq_host
    done

    read -e -p " * Please specify port of custom RabbitMQ server: " -i "$custom_rabbitmq_port" custom_rabbitmq_port
    while [ -z $custom_rabbitmq_port ]; do
        read -e -p " * Please specify port of custom RabbitMQ server: " -i "$custom_rabbitmq_port" custom_rabbitmq_port
    done

    echo -e "$custom_rabbitmq_username\n$custom_rabbitmq_passwd\n$custom_rabbitmq_host\n$custom_rabbitmq_port\n" > $rabbitmq_cust_cfg_file

    local rabbitmq_conf_pattern="\[rabbitmq\]\nhost=[^\n]*\nport=[^\n]*\nusername=[^\n]*\npassword=[^\n]*\n"
    local rabbitmq_conf_replacement="[rabbitmq]\nhost=$(escape_slash $custom_rabbitmq_host)\nport=$(escape_slash $custom_rabbitmq_port)\nusername=$(escape_slash $custom_rabbitmq_username)\npassword=$(escape_slash $custom_rabbitmq_passwd)\n"
    find /opt/caton/cydex3/ -type f -wholename */archive_*/config.ini -exec sed -i -z -re "s/$rabbitmq_conf_pattern/$rabbitmq_conf_replacement/g" {} \;
}

has_local_rabbitmq_server() {
    [ `rpm -qa | grep rabbitmq | wc -l` -gt 0 ]
}

check_need_install_local_rabbitmq() {
    # return false if explicitly indicated using external resource
    if [ "$1" == "1" ] ; then
        return 1
    fi
    if [ ! has_local_rabbitmq_server ] && [ -f $rabbitmq_cust_cfg_file ]; then
        echo "Skip installing local RabbitMQ by default, since custom defined RabbitMQ found in $rabbitmq_cust_cfg_file."
        read -e -p "Would you like to install local RabbitMQ server anyway? (y/N)" answer
        if [ "$answer" == "Y" ] || [ "$answer" == "y" ]; then
            return 0
        fi
        return 1
    fi
    return 0
}
