#!/bin/bash

show_usage() {
    echo "To backup or restore a one-server-deployed Cydex with local MySQL server (and local MongoDB if used)"
    echo -e "Usage:\n    $(basename $0) [backup|restore] [BACKUP_FOLDER_PATH]\n"
    echo "Example:"
    echo "    $0 backup /path/to/backup/folder"
    echo "    $0 restore /path/to/backup/folder"
    echo ""
    exit 1
}

if [[ $# -ne 2 ]]; then
    show_usage
fi
command=$1
if [[ "$command" -ne "backup" ]] && [[ "$command" -ne "restore" ]]; then
    show_usage
fi
backup_folder_path=$2
if [[ ! -d $backup_folder_path ]] && [[ -f $backup_folder_path ]]; then
    echo "Path $backup_folder_path already exists as a file"
    exit 1
fi
if [[ -d $backup_folder_path ]] && [[ $command == "backup" ]]; then
    read -e -p "Directory $backup_folder_path already exists. Do you want to overwrite it? (y/N) " answer
    if [[ "$answer" != "Y" ]] && [[ "$answer" != "y" ]]; then
        echo "Abort"
        exit 2
    fi
fi
if [[ ! -d $backup_folder_path ]]; then
    if [[ $command == "restore" ]]; then
        echo "Directory $backup_folder_path not exits"
        exit 1
    else
        mkdir -p $backup_folder_path
    fi
fi

source scripts/resource_funcs.sh
load_cust_mysql_vars
mysql_backup_path=$(realpath $backup_folder_path)/mysqlbackup.sql
unset mongo_dsn
unset mongo_backup_path
if has_local_mongo_server; then
    mongo_dsn=$(load_cust_mongo)
    mongo_backup_path=$(realpath $backup_folder_path)/mongobackup
fi
conf_backup_folder_path=$(realpath $backup_folder_path)/confs

backup() {
    # back up MySQL
    mysqldump -u$custom_mysql_username -p$custom_mysql_passwd --all-databases > $mysql_backup_path

    # back up MongoDB
    if [ ! -z $mongo_dsn ]; then
        mongodump --uri="$mongo_dsn" --archive=$mongo_backup_path
    fi

    # back up config
    if [ -f /opt/caton/cydex3/node/config.ini ]; then
        mkdir -p $conf_backup_folder_path/node
        cp -f /opt/caton/cydex3/node/config.ini $conf_backup_folder_path/node/
        if [ -f /opt/caton/cydex3/node/account.ini ]; then
            cp -f /opt/caton/cydex3/node/account.ini $conf_backup_folder_path/node/
        fi
    fi
    if [ -f /opt/caton/cydex3/archive_consumer/config.ini ]; then
        mkdir -p $conf_backup_folder_path/archive_consumer
        cp -f /opt/caton/cydex3/archive_consumer/config.ini $conf_backup_folder_path/archive_consumer/
    fi
}

restore() {
    # restore MySQL
    mysql -u$custom_mysql_username -p$custom_mysql_passwd < $mysql_backup_path

    # restore node account
    local node_id=$(grep -oP "^ *node_id *= *\K(.*)$" $conf_backup_folder_path/node/account.ini)
    local node_pw=$(grep -oP "^ *password *= *\K(.*)$" $conf_backup_folder_path/node/account.ini)
    if [[ -z "$node_id" ]] || [[ -z "$node_pw" ]]; then
        node_id=$(grep -oP "^ *node_id *= *\K(.*)$" $conf_backup_folder_path/node/config.ini)
        node_pw=$(grep -oP "^ *password *= *\K(.*)$" $conf_backup_folder_path/node/config.ini)
    fi
    if [[ -z "$node_id" ]] || [[ -z "$node_pw" ]]; then
        node_info_line=$(mysql -u$custom_mysql_username -p$custom_mysql_passwd -s -N db_zone -e "select id, password from node where enable = 1 order by last_login_at desc limit 1;")
        if [ ! -z "$node_info_line" ]; then
            node_vars=( $node_info_line )
            node_id=${node_vars[0]}
            node_pw=${node_vars[1]}
        fi
    fi
    find /opt/caton/cydex3/node/ -type f \( -name *.ini ! -name profile.ini \) -exec sed -i -zre 's/\[account\]\nnode_id *=[^\n]*\npassword *=[^\n]*\n*//g' {} \;
    echo -e "[account]\nnode_id = $node_id\npassword = $node_pw\n" >> /opt/caton/cydex3/node/account.ini

    # restore MongoDB
    if [ ! -z $mongo_dsn ]; then
        mongorestore --drop --uri="$mongo_dsn" --archive=$mongo_backup_path
    fi

    # reminder for configuration consistency
    echo "Restore finished. Please make sure your mount & archive configurations are consistent between current server and backup server. You may check files under folder $conf_backup_folder_path for reference."
}

sh scripts/stop_cydex.sh
eval "$command"
sh scripts/start_cydex.sh -m "all-archive"
